<?php

namespace App\Services;

use App\Models\Rider;
use App\Models\UserDepositBank;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\File;

use Carbon\Carbon;

use App\Models\User;

class PaystackService
{

    private $headers;
    private $base_url;

    public function __construct()
    {
        $this->headers = [
            'accept' => 'application/json',
            'authorization' => env("PAYSTACK_MODE") == 'test' ?
                'Bearer ' . env('PAYSTACK_TEST_SK') : 'Bearer ' . env('PAYSTACK_LIVE_SK'),
            'content-type' => 'application/json',
        ];

        $this->base_url = "https://api.paystack.co";
    }

    public function create_customer(User $user): bool
    {
        try {
            $response = Http::withHeaders($this->headers)->post("$this->base_url/customer", [
                'email' => $user->email,
                'first_name' => $user->firstname,
                'last_name' => $user->lastname,
                'phone' => $user->mobile,
            ]);

            $rr = json_decode($response->body());

            if ($rr->status) {
                $user->paystack_customer_code = $rr->data->customer_code;
                $user->save();
                return true;
            } else {
                return false;
            }
        } catch (\Throwable $e) {
            report($e);
            return false;
        }
    }

    public function create_virtual_account(User $user): bool
    {
        try {
            $response = Http::withHeaders($this->headers)->post("$this->base_url/dedicated_account", [
                'customer' => $user->paystack_customer_code,
            ]);

            $rr = json_decode($response->body());

            if ($rr->status) {
                $b = new UserDepositBank();
                $b->user_id = $user->id;
                $b->bank_name = $rr->data->bank->name;
                $b->bank_code = $rr->data->bank->id;
                $b->account_name = $rr->data->account_name;
                $b->account_number = $rr->data->account_number;
                $b->save();

                return true;
            } else {
                return false;
            }
        } catch (\Throwable $e) {
            report($e);
            return false;
        }
    }

    public function resolve_account_number($bank_code, $account_number)
    {
        try {
            $response = Http::withHeaders($this->headers)->get("$this->base_url/bank/resolve", [
                'account_number' => $account_number,
                'bank_code' => $bank_code,
            ]);

            $rr = json_decode($response->body());

            if ($rr->status) {
                return $rr->data;
            } else {
                return false;
            }
        } catch (\Throwable $e) {
            report($e);
            return false;
        }
    }

    public function register_transfer_receipt(Rider $rider, $bank)
    {
        try {
            $response = Http::withHeaders($this->headers)->post("$this->base_url/transferrecipient", [
                "type" => "nuban",
                "name" => $bank->account_name,
                "account_number" => $bank->account_number,
                "bank_code" => $bank->bank_code,
                "currency" => "NGN",
                "description" => "bank for $rider->uuid"
            ]);

            $rr = json_decode($response->body());

            if ($rr->status) {
                return $rr->data->recipient_code;
            } else {
                return false;
            }
        } catch (\Throwable $e) {
            report($e);
            return false;
        }
    }

    public function withdraw_rider_funds(Rider $rider, $bank, $amount)
    {
        try {
            $recipent = $this->register_transfer_receipt($rider, $bank);

            if ($recipent) {
              $amount = $amount * 100;
              $pamount = (int) $amount;
              
                $response = Http::withHeaders($this->headers)->post("$this->base_url/transfer", [
                    'source' => 'balance',
                    'reason' => "transfer for $rider->uuid",
                    'amount' => $pamount,
                    'recipient' => $recipent,
                ]);

                $date = date("Y-m-d H:i:s");
                File::append(public_path() . '/static/webhook.txt', "=============================\n\n rider withdraw $date $pamount" . $response->body() . PHP_EOL . "\n=============================");

                $rr = json_decode($response->body());

                if ($rr->status) {
                    return $rr->data->transfer_code;
                } else {
                    return false;
                }
            } else {
                return false;
            }
        } catch (\Throwable $e) {
            report($e);
            return false;
        }
    }

    public function verify_transfer($transfer_code)
    {
        try {
            $response = Http::withHeaders($this->headers)->get("$this->base_url/transfer/$transfer_code", []);

            $rr = json_decode($response->body());

            if ($rr->status) {
                // dd($rr->data);
                if ($rr->data->status == "success") {
                    return true;
                } else {
                    return false;
                }
            } else {
                return false;
            }
        } catch (\Throwable $e) {
            report($e);
            return false;
        }
    }
}
